package S22;

import java.util.*;

import S20.NineSquares;

/**
 *  This class solves 3 types of search problems:
 *
 *    1. breadth first (no heursitics)
 *    2. A* using the count heuristic (credit for tiles "in place")
 *    3. A* using the Manhattan distance heuristic (distance of tiles out of place)
 */
public class BreadthFirst {
    static final public int BREADTH_FIRST = 1;
    static final public int A_STAR_MANHATTAN = 2;
    static final public int A_STAR_COUNT = 3;

    static public int DEBUG = 0;  // 0=no debug, 1=little debug, 2=lots of debug

    private Hashtable avoidDuplicates = new Hashtable();
    private NineSquares currentGoal;
    
    static public void main(String[] args) {
        BreadthFirst p = new BreadthFirst();
        //   tests for basic breadth first (no heuristics):
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("1238 4765"), BREADTH_FIRST);
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("617342508"), BREADTH_FIRST); // trivial test

        //   tests for A* manhatten:
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("1238 4765"), A_STAR_MANHATTAN);
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("617342508"), A_STAR_MANHATTAN); // trivial test
        boolean success = p.solve(new NineSquares("21 364875"), new NineSquares("12345678 "), A_STAR_MANHATTAN); // trivial test

        //   tests for A* count:
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("1238 4765"), A_STAR_COUNT);
        //boolean success = p.solve(new NineSquares("6173 4582"), new NineSquares("617342508"), A_STAR_COUNT); // trivial test
        System.out.println("Success of search: " + success);
    }

    /**
     *  Given one of the 3 strategies, attempt to solve the 8-puzzle problem. This
     * is the top level method for the search.
     */
    public boolean solve(NineSquares start, NineSquares goal, int strategy) {
        currentGoal = goal;  // for use in method expand
        LinkedList queue = new LinkedList();
        int countDuplicates = 0;
        int countExpanded = 0;
        // add initial nodes:
        queue.addLast(expand(strategy, start, start.possibleMoves(), new Vector()));
        while (true) {
            if (queue.isEmpty()) break;
            // debug:
            //if (DEBUG > 0) queue.debug();
            State nextState = (State) queue.removeFirst();
            // breadth first, so first check to see if goal is reached:
            if (nextState.currentPosition.equals(goal)) {
                System.out.println(" Goal met for board: " + nextState.currentPosition);
                System.out.println("Number of duplicate board positions avoided = " + countDuplicates);
                System.out.println("Number of expanded nodes (not including duplicates) = " + countExpanded);
                printMoveHistory(nextState);
                return true;
            }
            // now, check each possible move in currentState to see if it meets the goal (if not, add to end of work queue)
            for (int i = 0, size = nextState.moveList.size(); i < size; i++) {
                int move = ((Integer) nextState.moveList.elementAt(i)).intValue();
                NineSquares b = NineSquares.makeMove(nextState.currentPosition, move);
                String s2 = b.toString();
                if (avoidDuplicates.get(s2) != null) {
                    countDuplicates++;
                    continue; // already looked at this position, so skip it
                }
                countExpanded++;
                avoidDuplicates.put(s2, new Boolean(true));
                Vector possibleMoves = b.possibleMoves();
                Vector v2 = new Vector(nextState.moveHistory);
                v2.add(new Integer(move));
                State state = expand(strategy, b, possibleMoves, v2);
                queue.addLast(state);
            }
        }
        return false;
    }



    /**
     *  Utility to print out the move history for solving a puzzle.
     */
    public void printMoveHistory(State s) {
        Vector moves = s.moveHistory;
        System.out.print("\nMove history to goal:");
        for (int i = 0, size = moves.size(); i < size; i++) {
            System.out.print(" " + moves.elementAt(i));
        }
        System.out.println("\nTotal number of moves=" + moves.size());
    }

    /**
      * This method handles breadth first and both A* algorithms
      */
    public State expand(int strategy, NineSquares board, Vector possibleMoves, Vector oldMoveHistory) { // possibleMoves contains Integer objects
        State s = null;
        Vector scores;
        int num;
        switch (strategy) {
            case BREADTH_FIRST:
                s = new State(board, possibleMoves);
                break;
            case A_STAR_COUNT:
                // here, we will sort the possible moves by the euclidean distance:
                scores = new Vector(possibleMoves.size());
                for (int i=0, size=possibleMoves.size(); i<size; i++) {
                    // use COUNT
                    NineSquares b2 = NineSquares.makeMove(board, ((Integer)possibleMoves.elementAt(i)).intValue());
                    int score = count(b2, currentGoal); // currentGoal is a class variable
                    scores.add(new Integer(score));
                }
                // since there will be at most 4 possible moves, a "bubble sort" is efficient:
                num = scores.size();
                for (int i=0; i<(num - 1); i++) {
                    for (int j=i; j<num; j++) {
                        int s1 = ((Integer)scores.elementAt(i)).intValue();
                        int s2 = ((Integer)scores.elementAt(j)).intValue();
                        //System.out.println("s1="+s1+", s2="+s2);
                        if (s2 > s1) { // swap values
                            Object temp = scores.elementAt(i);
                            scores.setElementAt(scores.elementAt(j), i);
                            scores.setElementAt(temp, j);
                            temp = possibleMoves.elementAt(i);
                            possibleMoves.setElementAt(possibleMoves.elementAt(j), i);
                            possibleMoves.setElementAt(temp, j);
                        }
                    }
                }
                s = new State(board, possibleMoves);
                break;
            case A_STAR_MANHATTAN:
                // here, we will sort the possible moves by the manhatten distance:
                scores = new Vector(possibleMoves.size());
                for (int i=0, size=possibleMoves.size(); i<size; i++) {
                    // use MANHATTAN
                    NineSquares b2 = NineSquares.makeMove(board, ((Integer)possibleMoves.elementAt(i)).intValue());
                    int score = manhattan(b2, currentGoal); // currentGoal is a class variable
                    scores.add(new Integer(score));
                }
                // since there will be at most 4 possible moves, a "bubble sort" is efficient:
                num = scores.size();
                for (int i=0; i<(num - 1); i++) {
                    for (int j=i; j<num; j++) {
                        int s1 = ((Integer)scores.elementAt(i)).intValue();
                        int s2 = ((Integer)scores.elementAt(j)).intValue();
                        //System.out.println("s1="+s1+", s2="+s2);
                        if (s2 > s1) { // swap values
                            Object temp = scores.elementAt(i);
                            scores.setElementAt(scores.elementAt(j), i);
                            scores.setElementAt(temp, j);
                            temp = possibleMoves.elementAt(i);
                            possibleMoves.setElementAt(possibleMoves.elementAt(j), i);
                            possibleMoves.setElementAt(temp, j);
                        }
                    }
                }
                s = new State(board, possibleMoves);
                break;
        }
        for (int i = 0, size = oldMoveHistory.size(); i < size; i++) {
            s.moveHistory.add(oldMoveHistory.elementAt(i));
        }
        //System.out.println("possible moves: " + s);
        return s;
    }
    /**
     *  COUNT heuristic for A*
     */
    public int count(NineSquares b1, NineSquares b2) {
        int score = 0;
        for (int i=0; i<9; i++) {
            if (b1.b[i] == b2.b[i])  score++;
        }
        return score;
    }

    /**
     *  MANHATTAN heuristic for A*
     */
    public int manhattan(NineSquares b1, NineSquares b2) {
        int score = 1000;
        for (int i=0; i<9; i++) {
            // find index of cell # i in b2:
            int index = 0;
            for (int j=0; j<9; j++) {
                if (b1.b[i] == b2.b[j]) {
                    index = j;
                    break;
                }
            }
            score -= m_distance(i, index);
        }
        return score;
    }
    // the manhatten distance is: absoluteValue(row1 - row2) + absoluteValue(col1 - col2):
    private int m_distance(int index1, int index2) {
        int [] row = {0, 0, 0, 1, 1, 1, 2, 2, 2};
        int [] col = {0, 1, 2, 0, 1, 2, 0, 1, 2};
        return iabs(row[index1] - row[index2]) + iabs(col[index1] - col[index2]);
    }
    private int iabs(int i) {
        if (i > -1) return i;
        return -i;
    }


}


//    utility class: State (state of a board position with possible moves

class State {
    public NineSquares currentPosition = null;
    public Vector moveList = null;
    public Vector moveHistory = new Vector();

    public State(NineSquares position, Vector moveList) {
        currentPosition = position;
        this.moveList = moveList;
    }

    public String toString() { // printable representation of this object
        StringBuffer sb = new StringBuffer("State object " + ":");
        for (int i = 0, size = currentPosition.b.length; i < size; i++) {
            sb.append(" " + currentPosition.b[i]);
        }
        sb.append(" moves:");
        for (int i = 0, size = moveList.size(); i < size; i++) {
            sb.append(" " + moveList.elementAt(i));
        }
        return sb.toString();
    }
}
