// File: tneural.cxx
//
// Description: This is the source file for the TNeuralMatch C++ class.
//              This class provides both back propagation training and
//              runtime modules.
//
// Copyright 1991 by Mark Watson.  All rights reserved.
//

#include "applib.h"
#include "tneural.h"

extern "C" { double exp(double); };

static long seed = 123456789;

void Warning(char *);

static double random(double lower, double upper)
{
    seed = (seed * (long)13) / (long)11;
    long ival = seed % 1000;
    double x = (double)ival;
    return ((x * 0.001 * (upper - lower)) - lower);
}

static double sigmoid(double x)
{
    return (1.0 / (1.0 + exp(-x)));
}

static double sigmoidP(double x)
{
    double y = sigmoid(x);
    return (y * (1.0 - y));
}

static void check_allocation(char *prompt, double *p)
{
    if (p == (double *)NULL) {
        char buf[100];
        sprintf(buf,"Failed memory allocation for %s", prompt);
        Warning(buf);
    }
}

TNeuralNet::TNeuralNet(int a_input_size, int a_hidden_size, int a_output_size)
{
    input_size = a_input_size;
    hidden_size = a_hidden_size;
    output_size = a_output_size;

    check_allocation("input a", (input_activations = new double[input_size]) );
    check_allocation("hidden a", (hidden_activations = new double[hidden_size]) );
    check_allocation("output a", (output_activations = new double[output_size]) );
    
    check_allocation("hidden sums", (hidden_sums = new double[hidden_size]) );
    check_allocation("output sums", (output_sums = new double[output_size]) );
    
    check_allocation("hidden errors", (hidden_errors = new double[hidden_size]) );
    check_allocation("output errors", (output_errors = new double[output_size]) );
    
    check_allocation("w1", (w1 = new double[input_size*hidden_size]) );
    check_allocation("w2", (w2 = new double[hidden_size*output_size]) );
    for (int h=0; h<hidden_size; h++) {
        for (int i=0; i<input_size; i++) w1[i*hidden_size + h] = random(-0.2,0.2);
        for (int o=0; o<output_size; o++) w2[h*output_size + o] = random(-0.2,0.2);
    }
    input_samples = 0;
    output_samples = 0;
}

double TNeuralNet::auto_train(int max_iterations,int num_samples,
                            double *packed_inputs, double *packed_outputs)
{   double error = 0;
    for (int i=0; i<max_iterations; i++) {
        for (int s=0; s<num_samples; s++) {
            backwards_error_pass(&(packed_inputs[s*input_size]),
                                 &(packed_outputs[s*output_size]));
            for (int j=0; j<output_size; j++)
                error += output_errors[j] * output_errors[j];
        }
    }
    return error;
}

void TNeuralNet::recall(double *input_pattern, double *output_classification_vector)
{
}

TNeuralNet::~TNeuralNet()
{
    delete input_activations;
    delete hidden_activations;
    delete output_activations;
    delete hidden_sums;
    delete output_sums;
    delete w1;
    delete w2;
    if (input_samples != 0)  delete input_samples;
    if (output_samples!= 0)  delete output_samples;
}

void TNeuralNet::save(char *filename)
{
    filebuf out_file;
    if (out_file.open(filename, output)==0) {
        Warning("Could not open output file");
    }
    ostream out_stream(&out_file);
    // Read in header information from file to get the size of this neural network:
    out_stream << input_size << "\n";
    out_stream << hidden_size << "\n";
    out_stream << output_size << "\n";

    int j;
    for (int i=0; i<input_size; i++)
        for (j=0; j<hidden_size; j++)
            out_stream << w1[i*hidden_size+j] << "\n";

    for (i=0; i<hidden_size; i++)
        for (j=0; j<output_size; j++)
            out_stream << w2[i*output_size+j] << "\n";
            
    out_file.close();

}

void TNeuralNet::restore(char *filename)
{   int f_input_size, f_hidden_size, f_output_size;
    filebuf in_file;
    if (in_file.open(filename, input)==0) {
        Warning("Could not open input file");
    }
    istream in_stream(&in_file);
    // Read in header information from file to get the size of this neural network:
    in_stream >> f_input_size;
    in_stream >> f_hidden_size;
    in_stream >> f_output_size;
    if (f_input_size != input_size || f_hidden_size != hidden_size || f_output_size != output_size) {
        Warning("Mismatch between current network size and training file.");
    }

    int j;
    for (int i=0; i<input_size; i++)
        for (j=0; j<hidden_size; j++) {
            in_stream >> w1[i*hidden_size+j];
        }

    for (i=0; i<hidden_size; i++)
        for (j=0; j<output_size; j++) {
            in_stream >> w2[i*output_size+j];
        }
            
    in_file.close();
}

void TNeuralNet::forward_pass(double *inputs)
{
    for (int i=0; i<input_size; i++)
        input_activations[i] = inputs[i];
    for (int h=0; h<hidden_size; h++) {
        hidden_sums[h] = 0.0;
        for (i=0; i<input_size; i++)
            hidden_sums[h] += input_activations[i] * w1[i*hidden_size + h];
    }
    for (h=0; h<hidden_size; h++)  hidden_activations[h] = sigmoid(hidden_sums[h]);
    for (int o=0; o<output_size; o++) {
        output_sums[o] = 0.0;
        for (h=0; h<hidden_size; h++)
            output_sums[o] += hidden_activations[h] * w2[h*output_size + o];
    }
    for (o=0; o<output_size; o++) output_activations[o] = sigmoid(output_sums[o]);
}

void TNeuralNet::backwards_error_pass(double *inputs, double *target_outputs)
{
    forward_pass(inputs);
    for (int o=0; o<output_size; o++)
        output_errors[o]=(target_outputs[o]-output_activations[o])*sigmoidP(output_sums[o]);
    for (int h=0; h<hidden_size; h++) {
        hidden_errors[h] = 0.0;
        for (o=0; o<output_size; o++)
            hidden_errors[h] += output_errors[o] * w2[h*output_size+o];
    }
    for (h=0; h<hidden_size; h++)
        hidden_errors[h] = hidden_errors[h] * sigmoidP(hidden_sums[h]);
    for (o=0; o<output_size; o++)
        for (h=0; h<hidden_size; h++)
            w2[h*output_size+o] += 0.4*output_errors[o]*hidden_activations[h];
    for (h=0; h<hidden_size; h++)
        for (int i=0; i<input_size; i++)
            w1[i*hidden_size+h] += 0.4*hidden_errors[h]*input_activations[i];
}

// Test code:

#if 0

#include <stdio.h>
#include <stdlib.h>

void Warning(char *w)
{
    cerr << w << "\n";
}

double Sinputs[9] = {0.9,0.1,0.1, 0.1,0.9,0.1, 0.1,0.1,0.9};
double Soutputs[9]= {0.1,0.9,0.1, 0.1,0.1,0.9, 0.9,0.1,0.1};

void main()
{
    cerr << "Starting test.\n";
    TNeuralNet nn(3,3,3);
//  nn.set_training_data(3, Sinputs, Soutputs);
//  nn.restore("test.dat");
    for (int j=0; j<20; j++) {
        double error = nn.auto_train(30,3,Sinputs,Soutputs);
        cerr << "Error: " << error << "\n";
    }
    nn.save("test.dat");
    cerr << "Done with test.\n";

}

#endif
