// File: TextInd.h
//
// Description: Header file for the text indexing class
//
// Copyright 1991 by Mark Watson.  All rights reserved
//

#include "char_v.h"

// Change the following 3 constant definitions to allow fewer or more
// file and block indices:

const int MAX_FILE_NAMES       = 8;
const int BITS_FOR_FILE_INDEX  = 3;
const int BITS_FOR_BLOCK_INDEX = 12;

const BLOCK_SIZE = 512;

void free_strings(char **s);

class filePointer {
public:
    unsigned block_index   : BITS_FOR_BLOCK_INDEX;
    unsigned file_index    : BITS_FOR_FILE_INDEX;
    unsigned end_of_list   : 1;
};

class char_5_block {  // 5 bit character, 'A' = 0, .. 'Z' = 25, end-of-line = 26.
public:
    unsigned n0 : 5;
    unsigned n1 : 5;
    unsigned n2 : 5;
    unsigned n3 : 5;
    unsigned n4 : 5;
    unsigned n5 : 5;
    unsigned n6 : 5;
    unsigned n7 : 5;
    unsigned n8 : 5;
    int equal(char_5_block *b);
};

// When we build the index data structures, we will build the index in
// place in the namePointerSpace data area. This will involve shifting
// large blocks of memory around, but will make the code much simpler.
// In virtual memory systems, this may cause excessive page faulting.

class TextIndex {
public:
    char_vect namePointerSpace;  // this will grow dynamically
    long topNamePointerSpace;     // size of data in namePointerSpace
    long firstCharIndex[28]; // first character position in the name/pointer
                             // space for each letter of the alphabet
    char *file_names[MAX_FILE_NAMES];
    int number_of_files;

    // various utility methods:
    Long firstHitIndex(char *name); // index in name/pointer space of
                                    // the first file/file block pointer
    void decodeFilePointer(filePointer *filePtr,
                           int &last_pointer_flag,   // equals 0 for last entry
                           int &file_index,          // file index for this entry
                           int &file_block_index);   // file blocks size=4096
    void setFilePointer(filePointer *filePtr,
                        int last_pointer_flag,   // equals 0 for last entry
                        int file_index,          // file index for this entry
                        int file_block_index);   // file blocks size=4096

    // class utilities:
    TextIndex(char **fileList, int num_files);  // constructor for building index
    TextIndex(char *fileName);   // constructor for reloading an old index
   ~TextIndex();
    void buildIndex_helper(char *file_name, int file_number);
    void buildIndex(); 
    void addWordToIndex(char *word, int file_count, int block_count);

    // File I/O methods:
    int save(char *indexFileName);
    int restore(char *indexFileName);

    // For retrieval:
    Char **regionTextBlocks(char *name); // returns 0 if name not indexed
    int regionIndex(int occurence, int &file, int &block); // returns 0 if name not indexed
    void get_text(char *buffer, int buffer_size, int file, int block);
    
    // Utilities for converting between 5 bit compact character codes and
    // 8 bit character codes:
    void c_5_to_8(char_5_block *compactChars, Char *bufferFor8BitChars);
    void c_8_to_5(Char *eightBitChars, char_5_block *bufferForCompactChars);
    
    int convert_5_to_8(int bit_5);
    int convert_8_to_5(int bit_8);
    
    // Utility to print out the index table for debugging:
    void print_out();
};
