// File: applib.cxx
//
// Description: Implementation of the X Window GUI library. Both
//              the Athena and Motif toolkits are supported.
//
// Copyright 1992 by Mark Watson Associates
//
//       No binary rights reserved: this software library may be used in
//       compiled form without restrictions.  All source rights
//       reserved: Source code to the GUI library can not be distributed
//       (on bulletin boards, or as part of shareware or commercial products)
//       without written permission.
//

extern "C" {
#include <sys/dirent.h>
};

#include "applib.h"

static int num_menu_items;
static char *menu_titles[10];

#ifdef MOTIF
#include <Xm/Separator.h>
#include <Xm/CascadeB.h>
#include <Xm/List.h>
#endif

#ifndef MOTIF
const y_offset = 18; // Make room for the menu button
#else
const y_offset = 28; // Make room for the menu button
#endif

// If the following flag is set, then popup dialog boxes will
// automatically goaway (as if the user clicked the cancel
// button) if the mouse cursor leaves the popup window.  I
// usually leave this set to false.
#define POPUP_GOAWAY 0

extern TAppWindow currentAppWindow;
extern Widget dWidget;
extern mouse_data data;

TAppWindow *current_window;

#define DO_IDLE_WORK

static int work_count = 0;
int do_work()
{
#ifdef DO_IDLE_WORK
   if (work_count++ >1000) {
       work_count = 0;
       current_window->idle_proc();
   }
#endif
   return 0;
}

void set_leave_widget_callback(Widget, XtActionProc);

void Warning(char *message)
{
    currentAppWindow.show_info(message);
}

void do_menu(Widget /*w*/, int client_data, int /* call_data*/ )
{
    currentAppWindow.do_menu_action((int)client_data + 1);
}

void start_mouse(Widget, mouse_data *, XEvent *event)
{
    currentAppWindow.mouse_down(event->xbutton.x, event->xbutton.y - y_offset);
}

void track_mouse(Widget, mouse_data *, XEvent *event)
{
  currentAppWindow.mouse_move(event->xbutton.x, event->xbutton.y - y_offset);
}

void end_mouse(Widget, mouse_data *, XEvent *event)
{
  currentAppWindow.mouse_up(event->xbutton.x, event->xbutton.y - y_offset);
}

void draw_screen(Widget, caddr_t, caddr_t)
{
  XClearArea(XtDisplay(dWidget), XtWindow(dWidget), 0, 0, 1024, 1024, 0);
  currentAppWindow.update_display();
  if (currentAppWindow.bottom_text_clip > 0 &&
      (currentAppWindow.in_scrolling_text_mode == 1 || 
       currentAppWindow.redraw_both_text_and_graphics == 1)) {
      int next = currentAppWindow.current_ring_buffer_start;
      for (int i=0; i<(currentAppWindow.number_of_saved_lines-1); i++) {
          next++;
          if (next > (currentAppWindow.number_of_saved_lines - 1))  next = 0;
          currentAppWindow.plot_string(currentAppWindow.left_text_clip + 1,
                          currentAppWindow.bottom_text_clip- 2
                          - (20*(currentAppWindow.number_of_saved_lines-i-2)),
                          currentAppWindow.saved_text[next]);
      }
  }
}

//  Data and code for do_edit method:

#ifndef MOTIF
#define XtDialogGetValueString XawDialogGetValueString
#endif

static int done_do_edit;
static char edit_field[256];
static void do_edit_ok(Widget, Widget dialog, caddr_t)
{   char *cp;
#ifndef MOTIF
    cp = XtDialogGetValueString(dialog);
    fprintf(stderr,"do_edit_ok: cp=|%s|\n", cp);
    sprintf(edit_field,"%s", cp);
    // Note: do not free storage pointed to by cp !!
#else
    cp = XmTextGetString((Widget)dialog);
int len = strlen(cp);
printf("len=%d, edit_field=%s\n",len,edit_field);
    sprintf(edit_field,"%s", cp);
    free(cp);
#endif
    done_do_edit = 1;
}
static void do_edit_cancel(Widget, Widget, caddr_t)
{
    sprintf(edit_field,"          ");
    done_do_edit = 1;
}

extern WidgetClass shellWidgetClass;
#ifndef MOTIF
extern WidgetClass dialogWidgetClass;
#else
extern WidgetClass xmTextWidgetClass;
#endif

extern Widget global_form;

static int popup_active = 0;

void TAppWindow::do_edit(char * prompt, char *returned_string)
{   Arg args[10];

    if (popup_active == 1) {
        fprintf(stderr," already using a popup\n");
        return;  // already using a popup
    }
    popup_active = 1;

    fprintf(stderr,"do_edit: returned_string=|%s|\n", returned_string);


    int n = 0;
    XtSetArg(args[n], XtNlabel, "Popup"); n++;
#ifndef MOTIF
    Widget p_do_edit = XtCreatePopupShell("do_edit", shellWidgetClass,
                                          global_form, args, n);
#else
    n = 0;
    XtSetArg(args[n], XmNautoUnmanage, FALSE); n++;
    Widget p_do_edit = XmCreateBulletinBoardDialog(global_form, "Edit", 
                                                   args, n);
#endif

#ifdef MOTIF
    Widget form = XtCreateManagedWidget("p_do_edit", xmFormWidgetClass,
                                        p_do_edit, NULL, 0);
#else
    Widget form = XtCreateManagedWidget("p_do_edit", formWidgetClass,
                                        p_do_edit, NULL, 0);
#endif
    n = 0;

#ifndef MOTIF
    XtSetArg(args[n], XtNlabel, prompt); n++;
    XtSetArg(args[n], XtNvalue, returned_string); n++;
    XtSetArg(args[n], XtNwidth, 560); n++;
    Widget dialog = XtCreateManagedWidget("dialog",dialogWidgetClass,
                                         form, args, n);
    n = 0;
    XtSetArg(args[n], XtNy, 520); n++;
    XtSetArg(args[n], XtNx, 10); n++;
    Widget ok_button = XtCreateManagedWidget("      OK      ",
                                             commandWidgetClass,
                                             dialog, args, n);
    n = 0;
    XtSetArg(args[n], XtNy, 520); n++;
    XtSetArg(args[n], XtNx, 90); n++;
    Widget cancel_button = XtCreateManagedWidget("    Cancel    ",
                                                 commandWidgetClass,
                                                 dialog, args, n);
    XtAddCallback(ok_button,XtNcallback, (XtCallbackProc)do_edit_ok, 
                  (XtPointer)dialog);

    XtAddCallback(cancel_button,XtNcallback,
                  (XtCallbackProc)do_edit_cancel,(XtPointer)NULL);
#else
    XtSetArg(args[n], XtNlabel, prompt); n++;
    Widget label = XtCreateManagedWidget(prompt,xmLabelWidgetClass,
                                         p_do_edit, args, n);
    n = 0;
    XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
    XtSetArg(args[n], XmNwordWrap, TRUE); n++;
    XtSetArg(args[n], XmNrows, 10); n++;
    XtSetArg(args[n], XmNcolumns, 60); n++;
    XtSetArg(args[n], XmNheight, 90); n++;
    XtSetArg(args[n], XmNy, 20); n++;
    XtSetArg(args[n], XmNx, 10); n++;
    XtSetArg(args[n], XmNvalue, returned_string); n++;
    Widget dialog = XtCreateManagedWidget("dialog",xmTextWidgetClass,
                                         p_do_edit, args, n);
    n = 0;
    XtSetArg(args[n], XmNy, 120); n++;
    XtSetArg(args[n], XmNx, 10); n++;
    Widget ok_button = XtCreateManagedWidget("      OK      ",
                                             xmPushButtonWidgetClass,
                                             p_do_edit, args, n);
    XtAddCallback(ok_button, XmNactivateCallback,
                  (XtCallbackProc)do_edit_ok, (XtPointer)dialog);
    n = 0;
    XtSetArg(args[n], XmNy, 120); n++;
    XtSetArg(args[n], XmNx, 90); n++;
    Widget cancel_button = XtCreateManagedWidget("    Cancel    ",
                                                 xmPushButtonWidgetClass,
                                                 p_do_edit, args, n);
    XtAddCallback(cancel_button, XmNactivateCallback,
                  (XtCallbackProc)do_edit_cancel, (XtPointer)NULL);
#endif

#if POPUP_GOAWAY
        set_leave_widget_callback(p_do_edit, (XtActionProc)do_edit_cancel);
#endif

    fprintf(stderr," before managing the popup\n");
//#ifndef MOTIF
    XtManageChild(p_do_edit);
//#else
//    XtPopup(p_do_edit, XtGrabNone);
//#endif

    /* Warp pointer to edit field */
#ifndef MOTIF
    XWarpPointer(XtDisplay(dialog),None,XtWindow(dialog),
                 0, 0, 0, 0, 40, 40);
    XSync(XtDisplay(p_do_edit),False);
#endif

    done_do_edit = 0;
    while (done_do_edit == 0) {
        XEvent an_event;
        XtNextEvent(&an_event);
        XtDispatchEvent(&an_event);
    }
printf("before popdown\n");
#ifndef MOTIF
    XtPopdown(p_do_edit);
#else
    XtUnmanageChild(p_do_edit);
#endif
printf("after popdown\n");
    XtDestroyWidget(p_do_edit);
printf("after XtDestroyWidget\n");
    popup_active = 0; // allow other popups to be created

    // Copy the edited string back into the returned_string buffer
    // (the caller must have allocated sufficient space for this):
    sprintf(returned_string, "%s", edit_field);

}


//  Data and code for show_info method:

static int done_show_info;
static void show_info_ok(Widget, Widget, caddr_t)
{
    done_show_info = 1;
}

void TAppWindow::show_info(char * prompt)
{   Arg args[5];

    if (popup_active == 1)  return;
    popup_active = 1;

    int n = 0;
    XtSetArg(args[n], XtNlabel, "Popup"); n++;
    Widget p_show_info = XtCreatePopupShell("show_info", shellWidgetClass,
                                          global_form, args, n);

#ifndef MOTIF
    Widget form = XtCreateManagedWidget("form",formWidgetClass,p_show_info,
                                 NULL, 0);
#else
    Widget form = XtCreateManagedWidget("form", xmFormWidgetClass,
                                        p_show_info, NULL, 0);
#endif

    n = 0;
    XtSetArg(args[n], XtNy, 2); n++;
    XtSetArg(args[n], XtNlabel, prompt); n++;

#ifdef MOTIF
    Widget label = XtCreateManagedWidget(prompt,xmLabelWidgetClass,
                                          form, args, n);
#else
    Widget label = XtCreateManagedWidget(prompt,labelWidgetClass,
                                         form, args, n);
#endif

#ifdef MOTIF
    n = 0;
    XtSetArg(args[n], XmNy, 40); n++;
    Widget ok_button = XtCreateManagedWidget("OK",xmPushButtonWidgetClass,
                                             form, args, n);
    XtAddCallback(ok_button, XmNactivateCallback,
                  (XtCallbackProc)show_info_ok, (XtPointer)NULL);
#else
    n = 0;
    XtSetArg(args[n], XtNy, 110); n++;
    XtSetArg(args[n], XtNfromVert, label); n++;
    XtSetArg(args[n], XtNvertDistance, 20); n++;
    Widget ok_button = XtCreateManagedWidget("OK",commandWidgetClass,
                                             form, args, n);
    XtAddCallback(ok_button,XtNcallback, (XtCallbackProc)show_info_ok, 
                  (XtPointer)NULL);
#endif

    XtManageChild(p_show_info);

    /* Warp pointer to edit field */
#ifndef MOTIF
    XWarpPointer(XtDisplay(ok_button),None,XtWindow(ok_button),
                 0, 0, 0, 0, 10, 10);
    XSync(XtDisplay(ok_button),False);
#endif
 
    done_show_info = 0;
    while (done_show_info == 0) {
        XEvent an_event;
        XtNextEvent(&an_event);
        XtDispatchEvent(&an_event);
    }
    XtPopdown(p_show_info);
    XtDestroyWidget(p_show_info);
    popup_active = 0;
}

// Choose one of two choices:

static int done_choose_one;
static void choose_one_cb(Widget, int value, caddr_t)
{
    done_choose_one = value;
}

int TAppWindow::choose_one(char * prompt1, char * prompt2)
{   Arg args[5];

    if (popup_active == 1)  return 0;
    popup_active = 1;

    int n = 0;
    XtSetArg(args[n], XtNlabel, "Popup"); n++;
    Widget p_choose_one = XtCreatePopupShell("choose_one", shellWidgetClass,
                                          global_form, args, n);

#ifndef MOTIF
    Widget form = XtCreateManagedWidget("form",formWidgetClass,p_choose_one,
                                 NULL, 0);
#else
    Widget form = XtCreateManagedWidget("form",xmFormWidgetClass,p_choose_one,
                                 NULL, 0);
#endif

    n = 0;
#ifndef MOTIF
    XtSetArg(args[n], XtNvertDistance, 20); n++;
    Widget button_1 = XtCreateManagedWidget(prompt1,commandWidgetClass,
                                            form, args, n);
    XtAddCallback(button_1, XtNcallback, (XtCallbackProc)choose_one_cb, 
                  (XtPointer)1);
#else
    Widget button_1 = XtCreateManagedWidget(prompt1,xmPushButtonWidgetClass,
                                            form, args, n);
    XtAddCallback(button_1, XmNactivateCallback,
                  (XtCallbackProc)choose_one_cb, (XtPointer)1);
#endif

    n = 0;
#ifndef MOTIF
    XtSetArg(args[n], XtNfromVert, button_1); n++;
    XtSetArg(args[n], XtNvertDistance, 10); n++;
    Widget button_2 = XtCreateManagedWidget(prompt2,commandWidgetClass,
                                            form, args, n);
    XtAddCallback(button_2,XtNcallback, (XtCallbackProc)choose_one_cb, 
                  (XtPointer)2);
#else
    XtSetArg(args[n], XmNy, 40); n++;
    Widget button_2 = XtCreateManagedWidget(prompt2,xmPushButtonWidgetClass,
                                            form, args, n);
    XtAddCallback(button_2, XmNactivateCallback,
                  (XtCallbackProc)choose_one_cb, (XtPointer)2);
#endif

    XtManageChild(p_choose_one);

    /* Warp pointer to edit field */
#ifndef MOTIF
    XWarpPointer(XtDisplay(button_1),None,XtWindow(button_1),
                 0, 0, 0, 0, 10, 10);
    XSync(XtDisplay(button_1),False);
#endif
 
    done_choose_one = 0;
    while (done_choose_one == 0) {
        XEvent an_event;
        XtNextEvent(&an_event);
        XtDispatchEvent(&an_event);
    }
    XtPopdown(p_choose_one);
    XtDestroyWidget(p_choose_one);
    popup_active = 0;
    return done_choose_one;
}

// List selection:

static int done_list_choose;

static void list_cancel_cb(Widget, int, caddr_t)
{
    done_list_choose = -1;
}

#ifndef MOTIF
static void list_cb(Widget this_list, int, caddr_t)
{
    XawListReturnStruct *list_struct = XawListShowCurrent(this_list);
    done_list_choose = 1 + list_struct->list_index;
    free((char *)list_struct); // we need to give back this storage
}
#else
static void list_cb(Widget, int, XmListCallbackStruct *list)
{
    done_list_choose = 1 + list->item_position;
}
#endif

int TAppWindow::choose_one_from_list(char *prompt, char **list_items,
                                     int number_of_items)
{   Arg args[10];

    if (popup_active == 1)  return -1;
    popup_active = 1;

    int n = 0;
    XtSetArg(args[n], XtNlabel, "Popup"); n++;
    Widget p_list_choose = XtCreatePopupShell("list_choose", shellWidgetClass,
                                          global_form, args, n);

#ifndef MOTIF
    Widget form = XtCreateManagedWidget("form",formWidgetClass,p_list_choose,
                                 NULL, 0);
#else
    Widget form = XtCreateWidget("form",xmFormWidgetClass,p_list_choose,
                                 NULL, 0);
#endif
    n = 0;
#ifndef MOTIF
    XtSetArg(args[n], XtNy, 2); n++;
    XtSetArg(args[n], XtNlabel, prompt); n++;
    Widget label = XtCreateManagedWidget(prompt,labelWidgetClass,
                                         form, args, n);
#else
    XtSetArg(args[n], XmNy, 2); n++;
    Widget label = XtCreateManagedWidget(prompt,xmLabelWidgetClass,
                                         form, args, n);
#endif

    n = 0;

#ifndef MOTIF
    XtSetArg(args[n], XtNfromVert, label); n++;
    XtSetArg(args[n], XtNvertDistance, 10); n++;
    XtSetArg(args[n],XtNlist,list_items);  n++;
    XtSetArg(args[n],XtNnumberStrings,number_of_items);  n++;
    int num_columns = 1 + (number_of_items / 30);
    XtSetArg(args[n],XtNdefaultColumns,num_columns);  n++;
    XtSetArg(args[n],XtNforceColumns,True); n++;
    Widget a_list=XtCreateManagedWidget("a_list",listWidgetClass,form,args,n);
    XtAddCallback(a_list,XtNcallback,(XtCallbackProc)list_cb,(XtPointer)NULL);
#else
    XmString *string_MOTIF = (XmString *)XtMalloc(sizeof(XmString)
                                                  *(number_of_items + 1));
    for (int k=0; k<number_of_items; k++)
        string_MOTIF[k] = XmStringCreate(list_items[k],
                                         XmSTRING_DEFAULT_CHARSET);
printf("size of list=%d\n",number_of_items);
    XtSetArg(args[n],XmNitems,string_MOTIF);  n++;
    XtSetArg(args[n],XmNitemCount,number_of_items);  n++;
    XtSetArg(args[n],XmNvisibleItemCount,number_of_items);  n++;
    XtSetArg(args[n], XmNy, 45); n++;
    Widget a_list=XtCreateManagedWidget("a_list",xmListWidgetClass,
                                 form,args,n);
    XtAddCallback(a_list,XmNbrowseSelectionCallback,
                  (XtCallbackProc)list_cb,(XtPointer)NULL);
    for (k=0; k<number_of_items; k++)
        XmStringFree(string_MOTIF[k]);
    free((char *)string_MOTIF);
#endif

    n = 0;
#ifndef MOTIF
    XtSetArg(args[n], XtNfromVert, a_list); n++;
    XtSetArg(args[n], XtNvertDistance, 10); n++;
    Widget button_2 = XtCreateManagedWidget("Cancel",commandWidgetClass,
                                            form, args, n);
    XtAddCallback(button_2,XtNcallback, (XtCallbackProc)list_cancel_cb, 
                  (XtPointer)1);
#else
    XtSetArg(args[n], XmNy, 18); n++;
    Widget button_2 = XtCreateManagedWidget("Cancel",xmPushButtonWidgetClass,
                                            form, args, n);
    XtAddCallback(button_2, XmNactivateCallback,
                  (XtCallbackProc)list_cancel_cb, (XtPointer)1);
#endif

#ifdef MOTIF
    XtManageChild(a_list);
    XtManageChild(form);
#endif

    XtManageChild(p_list_choose);


    /* Warp pointer to edit field */
#ifndef MOTIF
    XWarpPointer(XtDisplay(a_list),None,XtWindow(a_list),
                 0, 0, 0, 0, 40, 40);
    XSync(XtDisplay(a_list),False);
#endif
 
    done_list_choose = 0;
    while (done_list_choose == 0) {
        XEvent an_event;
        XtNextEvent(&an_event);
        XtDispatchEvent(&an_event);
    }
    XtPopdown(p_list_choose);
    XtDestroyWidget(p_list_choose);
    if (done_list_choose>-1) done_list_choose--;
    popup_active = 0;
    return done_list_choose;
}

void TAppWindow::clear_display()
{
    // Erase area and generate a redraw/expose event:
    XClearArea(XtDisplay(dWidget), XtWindow(dWidget), 0, 0, 1024, 1024, 1);
}

void TAppWindow::plot_line(int x1, int y1, int x2, int y2)
{
  /*
   * Draw once to clear the previous line.
   */
  XDrawLine(XtDisplay(dWidget), XtWindow(dWidget), data.gc, 
            x1, y1+ y_offset, x2, y2+ y_offset);
}

void TAppWindow::plot_rect(int top, int right, int bottom, int left)
{
    plot_line(left, top, left, bottom);
    plot_line(left, top, right, top);
    plot_line(right, top, right, bottom);
    plot_line(right, bottom, left, bottom);
}

void TAppWindow::erase_rect(int top, int right, int bottom, int left)
{
    // Erase area without generating a redraw/expose event:
    XClearArea(XtDisplay(dWidget), XtWindow(dWidget),
               left, top + y_offset, right, bottom + y_offset, 0);
}

void TAppWindow::plot_string(int x, int y, char *str)
{
  /*
   * Draw once to clear the previous line.
   */
  XDrawString(XtDisplay(dWidget), XtWindow(dWidget), data.gc, 
            x, y+ y_offset, str, strlen(str));
}

// Utility functions for popup menus for ATHENA:

#ifndef MOTIF

static int x_location = 350;
static int y_location = 300;

static int  poped_up = 0;

#define XtPointer caddr_t

Widget popup_shell, popup_shell2;

extern "C" {
    void XtPopdown(Widget);
};

static int local_count_callbacks = 1;

typedef XtActionsRec aXtActionsRec;

void set_leave_widget_callback(Widget w, XtActionProc cb)
{
   static char buf[100], name[20];
   sprintf(name,"name00%d",local_count_callbacks++);
   sprintf(buf,"<LeaveNotify>: %s()",name);
   static aXtActionsRec myActions[1];
   myActions[0].proc = (XtActionProc)cb;
   myActions[0].string = name;
   XtAddActions(myActions,1);
   XtOverrideTranslations(w,XtParseTranslationTable(buf));
}

void set_enter_widget_callback(Widget w, XtActionProc cb)
{
   static char buf[100], name[20];
   sprintf(name,"name00%d",local_count_callbacks++);
   sprintf(buf,"<EnterNotify>: %s()",name);
   static aXtActionsRec myActions[1];
   myActions[0].proc = (XtActionProc)cb;
   myActions[0].string = name;
   XtAddActions(myActions,1);
   XtOverrideTranslations(w,XtParseTranslationTable(buf));
}

void PaneChosen(Widget w, XtPointer client_data, XtPointer)
{ 
    int selection_number = (int) client_data;
    do_menu(w, selection_number, 0);
    XtPopdown(popup_shell);
}

void do_popdown(Widget, caddr_t, caddr_t)
{
    XtPopdown(popup_shell);
    poped_up = 0;
}

void makeMenu(Widget topLevel, char **item_names, int num_items)
{   int i;
    Widget menubox, a_form, menupane[10];
    Arg args[10];
    unsigned int key_buttons;
    int win_x, win_y;
    Window root, child;

    if (poped_up == 1)   return;
    poped_up = 1;

    XQueryPointer(XtDisplay(topLevel),XtWindow(topLevel),
                  &root, &child, &x_location, &y_location, &win_x, &win_y,
                  &key_buttons);

    // Throw away all mouse movement events so the popup menu
    // does not popdown due to residual mouse motion:
    XSync(XtDisplay(topLevel), False);

    i=0;
    XtSetArg(args[i],XtNx,x_location-40); i++;
    XtSetArg(args[i],XtNy,y_location-40); i++;
    XtSetArg(args[i],XtNborderWidth,4); i++;

    popup_shell = XtCreatePopupShell("popup_shell", overrideShellWidgetClass,
                                     topLevel, args, i);

    i=0;
    XtSetArg(args[i],XtNborderWidth,1); i++;

    a_form = XtCreateManagedWidget("a_form", formWidgetClass,
                                    popup_shell, args, i);

    menubox = XtCreateManagedWidget("menubox", boxWidgetClass,
                                    a_form, args, i);

    for (i = 0; i < num_items; i++) {
        int j = 0;
        XtSetArg(args[j],XtNborderWidth,0); j++;
        XtSetValues(popup_shell,args,j);
        menupane[i] = XtCreateManagedWidget(item_names[i],
                                            commandWidgetClass, menubox,
                                            args, j);
        XtAddCallback(menupane[i], XtNcallback,
                      (XtCallbackProc)PaneChosen, (XtPointer)i);
    }
    set_leave_widget_callback(popup_shell, (XtActionProc)do_popdown);

    XtPopup(popup_shell,XtGrabNone);

   /* Warp pointer to edit field */
#ifndef MOTIF
   XWarpPointer(XtDisplay(popup_shell),None,XtWindow(popup_shell),
      0, 0, 0, 0, 40, 40);
   XSync(XtDisplay(topLevel),False);
#endif
}

#endif  // #ifndef MOTIF

#ifdef MOTIF

void makeMenu(Widget parent, char **item_names, int num_items)
{   Arg args[4];

    Widget the_menu =XmCreatePulldownMenu(parent,
                                   "Menu1",
                                   NULL, 0);
    XtSetArg(args[0], XmNsubMenuId, the_menu);
    Widget cascade = XtCreateWidget("Menu", xmCascadeButtonWidgetClass,
                                    parent,args,1);
    WidgetList menu_buttons = (WidgetList)XtMalloc(num_items*sizeof(Widget));
    for(int i=0;i<num_items;i++) {
        menu_buttons[i] = XtCreateWidget(item_names[i],
                                         xmPushButtonWidgetClass,
                                         the_menu, NULL, 0);
        XtAddCallback(menu_buttons[i], XmNactivateCallback,
                      (XtCallbackProc)do_menu,(XtPointer)i);
    }
    XtManageChildren(menu_buttons, num_items);
    XtManageChild(cascade);
}

#endif

void do_make_menu(Widget, int, int)
{
  makeMenu(global_form, menu_titles, num_menu_items);
}



// Create our standard XOR GC for drawing:

GC createXorGC(Widget w)
{
    GC        gc;
    XGCValues values;
    Arg       args[5];  // Leave room for future code modifications
    int n = 0;
    XtSetArg(args[n], XtNforeground, &values.foreground); n++;
    XtSetArg(args[n], XtNbackground, &values.background); n++;
    XtGetValues(w, args,n);
printf("foreground, background=%d  %d\n",values.foreground,values.background);
//    values.foreground = values.foreground ^ values.background;
    values.line_style = LineSolid;
    values.function   = GXxor;  // different than createCopyGC
    gc = XtGetGC(w, GCForeground | GCBackground | 
                 GCFunction | GCLineStyle, &values);
    return gc;
}

// Create our COPY GC for drawing:

GC createCopyGC(Widget w)
{
    GC gc;

    XGCValues values;
    Arg       args[5];  // Leave room for future code modifications
    int n = 0;
    XtSetArg(args[n], XtNforeground, &values.foreground); n++;
    XtSetArg(args[n], XtNbackground, &values.background); n++;
    XtGetValues(w, args,n);
printf("foreground, background=%d  %d\n",values.foreground,values.background);
//    values.foreground = values.foreground ^ values.background;
    values.line_style = LineSolid;
    values.function   = GXcopy;  // different than createXorDC
    gc = XtGetGC(w, GCForeground | GCBackground | 
                 GCFunction | GCLineStyle, &values);
    return gc;
}

// Support for crolling text:


void TAppWindow::init_scrolling_text(int top, int right, int bottom, int left)
{
    top_text_clip    = top;
    right_text_clip  = right;
    bottom_text_clip = bottom;
    left_text_clip   = left;
    number_of_saved_lines = -1 + ((bottom_text_clip - top_text_clip) /
                                 string_height(" "));
    if (number_of_saved_lines > MAX_TEXT_LINES)
       number_of_saved_lines = MAX_TEXT_LINES;
    if (number_of_saved_lines < 3) number_of_saved_lines = 3;
    XClearArea(XtDisplay(dWidget), XtWindow(dWidget), left,top,right,bottom, 0);

    current_ring_buffer_start = 0;
    // Allocate storage for saved text lines:
    for (int i=0; i<number_of_saved_lines; i++) {
        saved_text[i] = new char[MAX_TEXT_LINE_SIZE + 1];
        saved_text[i][0] = '\0';
    }
    in_scrolling_text_mode = 1;
    redraw_both_text_and_graphics = 1;
}

void TAppWindow::init_scrolling_text()
{   
    // Set the text area with content rectangle of the window:

    top_text_clip    = 0;
    right_text_clip  = 256;
    bottom_text_clip = 310;
    left_text_clip   = 0;

    number_of_saved_lines = -1 + ((bottom_text_clip - top_text_clip) /
                                 string_height(" "));
    if (number_of_saved_lines > MAX_TEXT_LINES)
       number_of_saved_lines = MAX_TEXT_LINES;
    if (number_of_saved_lines < 3) number_of_saved_lines = 3;

    XClearArea(XtDisplay(dWidget), XtWindow(dWidget), 0, 0, 1024, 1024, 0);
    
    current_ring_buffer_start = 0;
    // Allocate storage for saved text lines:
    for (int i=0; i<number_of_saved_lines; i++) {
        saved_text[i] = new char[MAX_TEXT_LINE_SIZE + 1];
        saved_text[i][0] = '\0';
    }
    in_scrolling_text_mode = 1;
    redraw_both_text_and_graphics = 0; // entire window used for text.
}

void TAppWindow::put_scrolling_text(char *str)
{
    if (bottom_text_clip != 0)  {
        XCopyArea(XtDisplay(dWidget),XtWindow(dWidget),XtWindow(dWidget),
                  data.gc, 0,50, // src x,y
                  256, 256, 0, 30);
        plot_string(left_text_clip + 1, bottom_text_clip - 2, str);
        char buf[257];
        if (strlen(str) < 256) {
            sprintf(buf,"%s",str);
            buf[MAX_TEXT_LINE_SIZE] = '\0';
            sprintf(saved_text[current_ring_buffer_start],"%s",buf);
            current_ring_buffer_start++;
            if (current_ring_buffer_start > (number_of_saved_lines - 1))
                current_ring_buffer_start = 0;
        }
    } else {
        Warning("Called TAppWindow::put_scrolling_text before TAppWindow::init_scrolling_text");
    }
    in_scrolling_text_mode = 1;
}


void TAppWindow::reset_scrolling_text()
{
    for (int i=0; i<MAX_TEXT_LINES; i++) {
        saved_text[i][0] = '\0';
    }
}

#if 0
extern "C" {
    int strlen(char *);
    int strcmp(char *, char *);
};
#endif

static int find_extension_in_file_name(char *file, char *extension)
{   char buf[64];
    sprintf(buf,".%s", extension);
    int len_file = strlen((char *)file);
    int len_ext  = strlen((char *)buf);
    for (int k=0; k<(len_file - len_ext + 1); k++) {
        if (strcmp((char *)buf, (char *)(&(file[k]))) == 0)
            return 1;
    }
    return 0;
}
 
int TAppWindow::choose_file_to_read(char *prompt, char *extension,
                                    char *file_name)
{
    int numFiles = 0;
    DIR *dirp;
    struct dirent *dp;
    char *fileNames[500];
    dirp = opendir(".");
    if (dirp != 0) {
        while ((dp = readdir(dirp)) != 0) {
            if (strlen(extension) < 1 || 
                find_extension_in_file_name(dp->d_name,extension))
                if (numFiles < 498) {
                    // Insert the file in the correct (sorted position):
                    int inserted = 0;
                    for (int m=0; m<numFiles; m++)
                        if (strcmp((char *)dp->d_name,
                                   (char *)fileNames[m])<=0 &&
                            inserted == 0) {
                            for (int k=(numFiles-1); k>=m; k--)
                                fileNames[k+1] = fileNames[k];
                            int len = strlen((char *)(dp->d_name));
                            fileNames[m] = new char[len + 1];
                            for (k=0; k<len; k++)
                                fileNames[m][k] = dp->d_name[k];
                            fileNames[m][len] = '\0';
                            numFiles++;
                            inserted = 1;
                        }
                    if (inserted == 0) {
                        int len = strlen((char *)(dp->d_name));
                        fileNames[numFiles] = new char[len + 1];
                        for (int k=0; k<len; k++)
                            fileNames[numFiles][k] = dp->d_name[k];
                        fileNames[numFiles][len] = '\0';
                        numFiles++;
                    }
                }
        }
    }
    int response = choose_one_from_list(prompt, fileNames, numFiles);
    if (response > -1) {
        sprintf(file_name,"%s",fileNames[response]);
    }  else {
        file_name[0] = '\0';
    }
    for (int k=0; k<numFiles; k++) delete fileNames[k];
    if (strlen(file_name) > 0) return 0;
     else                      return 1;
}

int TAppWindow::choose_file_to_write(char *prompt, char *file_name)
{
    do_edit(prompt, file_name);
    if (strlen(file_name) > 0) return 0;
     else                      return 1;
}

void set_menu(char *menu_title, int num_items, char **items)
{
    num_menu_items = num_items;
    for (int i=0; i<num_items; i++)
        menu_titles[i]=items[i];
}

